## ----init,include=FALSE,cache=FALSE-------------------------------------------
 opts_chunk[["set"]](fig.path='figure/graphics-', cache.path='cache/graphics-', fig.align='center', dev='tikz', external=FALSE, fig.width=4.5, fig.height=3, fig.show='hold', echo=FALSE,warning=FALSE, error=FALSE, message=FALSE, tidy=FALSE,cache=TRUE,autodep=TRUE, par=TRUE, comment=NA, keep.blank.line=FALSE, tidy=FALSE)
 knit_hooks[["set"]](par=function(before, options, envir){
 if (before) { options(width=80) }
 if (before && options[["fig.show"]]!='none') par(mar=c(4,4,0,.1),cex.main=1,font.main=1,cex.lab=.95,cex.axis=.9,mgp=c(2,.7,0),tcl=-.3)
 }, crop=hook_pdfcrop)
options(tikzMetricsDictionary="tikz.metrics")
options( tikzDocumentDeclaration = '\\documentclass[12pt]{scrartcl}\\usepackage[utf8]{inputenc}\\usepackage[tt=false]{libertine}\\usepackage[libertine]{newtxmath}\\usepackage{amsmath}\\input{paperDefs.tex}')


## ----cache=FALSE--------------------------------------------------------------
library(lattice)
library(latticeExtra)
library(coda)
library(runjags)
library(xtable)
library(lme4)
library(boot)
library(plyr)
dfSample <- function(df,size) { n<-nrow(df); df[sample(n,size),] }
simCols <- function(df,vars,n) {
    zz<-list()
    for(name in vars) zz[[name]]<- sample(c(df[,grep(name,colnames(df))]),n)
    as.data.frame(zz)
}

library(zTree)
initJags<-list()
initJags[[1]]<-list(.RNG.seed=1,.RNG.name="base::Mersenne-Twister")
initJags[[2]]<-list(.RNG.seed=2,.RNG.name="base::Super-Duper")
initJags[[3]]<-list(.RNG.seed=3,.RNG.name="base::Wichmann-Hill")
initJags[[4]]<-list(.RNG.seed=4,.RNG.name="lecuyer::RngStream")
jagsVer<-testjags(silent=TRUE)
#
mTheme<-standard.theme("pdf", color=FALSE)
lattice.options(default.theme=mTheme)
#
bsSize<-1000
jagsAdapt<-1000
jagsBurnin<-4000
jagsSample<-100000
jagsThin<-1
TIME=FALSE
#


## ----rawData,eval=FALSE-------------------------------------------------------
## setwd("raw")
## raw<-zTreeTables(dir(pattern="*.xls"))
## setwd("..")
## sub<-raw$subjects
## # get a subject ID which is `nice':
## sub<-within(sub,{
##     sid<-as.factor(sub("_",".",sprintf("%s-%s",Date,Subject)))
##     gid<-as.factor(sub("_",".",sprintf("%s-%s",Date,BigGroup)))
## })
## PGsubjects<-unique(subset(sub,Type<5)$sid)
## sub<-subset(sub,sid %in% PGsubjects) # restrict to subjects who participate in PG game
## sub<-within(sub,sid<-factor(sid,level=PGsubjects))
## #
## # only Treatment=2 is the public good treatment:
## pubGood<-subset(sub,Treatment==2)[,grep("sid|gid|input|Period|Type",
##                         names(sub),value=TRUE)]
## pubGood<-rename(pubGood,c("input"="contribution"))
## # find the right type of punishment:
## pubGoodL<-reshape(pubGood,varying=grep("input_T",names(pubGood)),sep="_T",
##                   direction="long")
## pubGoodL<-subset(pubGoodL,Type==time)
## pubGoodL<-rename(pubGoodL,c("input"="punrec"))
## # do the lags and diffs:
## pubGoodL<-pubGoodL[order(pubGoodL$sid,pubGoodL$Period),]
## pubGoodL<-within(pubGoodL,{
##     D.contrib<-c(NA,diff(contribution))
##     L.punrec<-c(NA,punrec[-length(punrec)])
##     L.Period<-c(NA,Period[-length(punrec)])
##     D.contrib[Period!=L.Period+1]<-NA
##     L.punrec[Period!=L.Period+1]<-NA
## })
## # Treatment 3 is the risk treatment and has "eins" defined:
## # keep the relevant part of the data:
## sub<-subset(sub,!is.na(eins))
## # translate the funny variable names...
## funnyNames<-c("eins","zwei","drei","vier","fuenf","sechs","sieben",
##               "acht","neun","zehn")
## sub1<-sub[,funnyNames]
## betterNames<-sprintf("y.%d",1:length(funnyNames))
## names(sub1)<-betterNames
## sub1$sid<-sub$sid
## # translate to long form:
## subL<-within(reshape(sub1,direction="long",idvar="sid",varying=betterNames,timevar="r"),r<-r/10)
## save(sub,subL,pubGood,pubGoodL,file="data/rawPrep.Rdata")

## ----critEst,cache=FALSE------------------------------------------------------
load("data/rawPrep.Rdata")
xcritMod <- 'model {
    for (i in 1:length(y)) {
        y[i] ~ dbern(p[i])
        logit(p[i]) <-  (r[i] - xcrit[sid[i]]) * pow(tauCrit[sid[i]],.5)
    }
    # subject specific effects:
    for (s in 1:max(sid)) {
        xcrit[s]   ~ dbeta(ac,bc)         # prior for switching point risk
        tauCrit[s] ~ dgamma(sG,rG)        # var switching point (prior below)
    }
    # contribution, heterogeneity:
    ac ~ dgamma(2,1/2)                       # prior for switching point
    bc ~ dgamma(2,1/2)                       # prior for switching point
    sG <- pow(m,2)/pow(d,2)               # prior for var switching point
    rG <- m/pow(d,2)                      # prior for var switching point
    m ~ dgamma(1,1)                       # prior for var switching point 
    d ~ dgamma(dalpha,dbeta)                       # prior for var switching point 
}'
#
# here is the ``prior'':
xcritNull <- 'model {
    for (s in 1:max(sid)) {
        xcrit[s]   ~ dbeta(ac,bc)         # prior for switching point risk
        tauCrit[s] ~ dgamma(sG,rG)        # var switching point (prior below)
    }
    # contribution, heterogeneity:
    ac ~ dgamma(2,1/2)                       # prior for switching point
    bc ~ dgamma(2,1/2)                       # prior for switching point
    sG <- pow(m,2)/pow(d,2)               # prior for var switching point
    rG <- m/pow(d,2)                      # prior for var switching point
    m ~ dgamma(1,1)                       # prior for var switching point 
    d ~ dgamma(dalpha,dbeta)                       # prior for var switching point 
}'
#
cData<-with(subL,list(y=y,r=r,sid=as.numeric(factor(sid))))
cData[["dalpha"]]<-10
cData[["dbeta"]]<-1
set.seed(123)
runjags.options(silent.jags=TRUE,silent.runjags=TRUE,method="parallel",
                modules=c("glm","lecuyer"),predraw.plots=FALSE,force.summary=TRUE)


## ----aggregateXcrit-----------------------------------------------------------
load("jjCrit.Rdata")
# we now have 10000 estimates of xcrit, aggregate so that we have one for 
# each subject (here we only want to identify the consistent subjects):
xxNull<-(combine.mcmc(jjNull))
xxCrit<-(combine.mcmc(jjCrit))
xxCritMedian<-apply(as.matrix(xxCrit),2,median)
# reshape results:
zz<-list()
for (n in c("tauCrit","xcrit")) {
    z<-xxCritMedian[grep(n,names(xxCritMedian))]
    i<-as.numeric(gsub("[^\\[]*\\[|\\]","",names(z)))
    zz[[n]]<-cbind(i,z)
}
critEst<-merge(zz[["tauCrit"]],zz[["xcrit"]],by="i")
names(critEst)<-c("sid","tau","xcrit")
save(critEst,file="data/critEst.Rdata")

## ----naiveSwitching,cache=TRUE------------------------------------------------
# add sid-factors to critEst
# also do the “naive” xcrit:
critEstF<-within(merge(within(critEst,{sid<-as.factor(sid);levels(sid)<-levels(subL$sid)}),
                       within(with(subset(subL,y==1),
    aggregate(r,list(sid=sid),min)),{xnaive<-x - 1/20;x<-NULL}),all.x=TRUE),{
        xnaive[is.na(xnaive)]<-1.05
    })
xxLogit<-as.data.frame(t(sapply(by(subL,list(subL[["sid"]]),function(x) {cc<-coef(glm(y ~ r,family=binomial,data=x,epsilon=1e-14,maxit=50));c(xLogit=pmin(1.05,-cc[1]/ifelse(abs(cc[2])<1e-8,abs(cc[2]),cc[2])),tauLogit=cc[2])}),c)))
xxLogit[["sid"]]<-factor(rownames(xxLogit),levels=levels(subL[["sid"]]))
names(xxLogit)[grep("tauLogit",names(xxLogit))]<-"tauLogit"
xxLogit<-merge(xxLogit,within(with(subL,aggregate(y ~ sid,FUN=sum)),{xcritHL<-(10.5-y)/10;y<-NULL}))
betaLogitHigh<-signif(min(abs(subset(xxLogit,abs(tauLogit)>40)[["tauLogit"]])),3)
betaLogitHighI<-signif(range(abs(subset(xxLogit,abs(tauLogit)>40)[["tauLogit"]])),3)
betaLogitLow<-subset(xxLogit,abs(tauLogit)<40)[["tauLogit"]]
critEstF<-merge(critEstF,xxLogit)
critEstF<-within(critEstF,consist<-tauLogit>100)
subLL<-merge(subL,critEstF)
subLL<-within(subLL,{sid<-reorder(sid,xcrit);nsid<-as.numeric(sid)})
subLL<-subLL[order(subLL[["nsid"]]),]
##
negTauLogit<-subset(with(subLL,aggregate(cbind(nsid=as.numeric(nsid),tauLogit)~sid,FUN=mean)),tauLogit< -.0001)
##
probSid<-unique(subset(subLL,!consist)[["nsid"]])
consistentShare<-mean(critEstF$consist)
#
LOGITCOUNTDIFFER<-paste(with(subLL,unique(as.numeric(sid[abs(xLogit-xcritHL)>.02]))),collapse=" and ")
LOGITCOUNTLOST<-paste(with(subLL,unique(as.numeric(sid[abs(xLogit-xcritHL)>10]))),collapse=" and ")


## ----jointData----------------------------------------------------------------
resDF <- function(df,ssid) {
    df2<-df
    df2$sid<-as.character(df2$sid)
    npack<-nrow(df)/length(ssid)
    for (i in 1:(length(ssid))) {
        r2<-((i-1)*npack+1):(i*npack)
        df2[r2,]<-subset(df,sid==ssid[i])
        df2[r2,"sid"]<-sprintf("%03d-%s",i,ssid[i])
    }
    df2$sid<-factor(df2$sid)
    df2
}
# prepare the data for all or only "consistent" cases:
mData <- function(allowIC,resample=FALSE) {
    pubGoodL2<-subset(pubGoodL,!is.na(L.punrec)) # <- restrict attention to relevant cases
    # resample critEstF(72), 
    if(resample) {
        ssid<-sample(unique(critEstF$sid),replace=TRUE)
        critEstF<-resDF(critEstF,ssid)
        pubGoodL2<-resDF(pubGoodL2,ssid)
        subL<-resDF(subL,ssid)
    }
    #
    subjects<-subset(critEstF,(consist | allowIC) & sid %in% unique(pubGoodL2)[["sid"]])[["sid"]]
    ssubL<-subset(subL,sid %in% subjects)
    PG <- subset(pubGoodL2,sid %in% subjects)
    PG<-merge(PG,critEstF[,c("sid","xLogit","xcritHL")],all.x=TRUE)
    sgid<-with(subset(pubGoodL2,sid %in% subjects),aggregate(gid,list(sid),function(x) x[1]))
    names(sgid)<-c("sid","gid")
    gid<-as.numeric(sgid[with(sgid,order(levels(sid)[sid])),"gid"])
    mData<-list()
    mData[["y"]]<-ssubL$y
    mData[["r"]]<-ssubL$r
    mData[["sid"]]<-as.numeric(factor(ssubL$sid,level=subjects))
    mData[["xnaive"]]<-critEstF[mData[["sid"]],"xnaive"]
    mData[["D.contrib"]]<-PG$D.contrib
    mData[["L.punrec"]]<-PG$L.punrec
    mData[["xLogit"]]<-PG$xLogit
    mData[["xcritHL"]]<-PG$xcritHL
    mData[["sidPG"]]<-as.numeric(factor(PG$sid,level=subjects))
    mData[["gid"]]<-as.numeric(factor(gid))
    mData[["gidPG"]]<-mData[["gid"]][mData[["sidPG"]]]
    mData[["Period"]]<-PG$Period
    mData
}
dataAll<-mData(allowIC=TRUE)
dataConsist<-mData(allowIC=FALSE)


## ----eval=FALSE---------------------------------------------------------------
## N<-100
## library(mvtnorm)
## set.seed(123)
## xx<-replicate(100,{
##     X<-rmvnorm(N,sigma=matrix(c(1,.5,.5,1),ncol=2))
##     Y<-X[,1]-X[,2]+rnorm(N)
##     xi<-X[,1]+rnorm(N)
##     coef(lm(Y ~ xi + X[,2]))
##     })
## plot(t(xx[-1,]))


## ----figChoicesNet,fig.height=2,fig.width=6.01--------------------------------
nn<-length(levels(subLL$sid))
#    
par(mar=c(4,4,0,0))
with(subset(subLL,y==1),plot(r ~ nsid,xlab="Participants",xaxt="n",ylab="$\\prob=P(\\text{good outcome})$",xlim=c(1,nn),ylim=c(.05,1.05),cex.lab=.8))
aa<-probSid[probSid%%2==0]
abline(v=probSid,lty="dotted")
axis(1,at=probSid,labels=FALSE)
axis(1,at=aa,tick=FALSE,cex.axis=.45)
aa<-probSid[probSid%%2!=0]
axis(1,at=aa,tick=FALSE,line=.5,cex.axis=.45)


## ----getTau-------------------------------------------------------------------
sdRange<-c(NA,NA)
if (file.exists("data/critEst.Rdata")) {
    load("data/critEst.Rdata")
    sdRange<-signif(with(critEst,range(1/sqrt(tau))),2)
}


## -----------------------------------------------------------------------------
xxTauNull<-xxNull[,grep("tauCrit",colnames(xxNull))]
xxTauCrit<-xxCrit[,grep("tauCrit",colnames(xxCrit))]
tauConsist<-quantile(critEst[["tau"]],1-consistentShare/2) # median consistent decision maker


## ----figEstSwitchPointL,fig.height=2.5,fig.width=6.01-------------------------
nn<-length(levels(subLL$sid))
layout(rbind(2,3,1),heights=c(.25,.05,.7))
#    
par(mar=c(4,4,0,0))
with(subset(subLL,y==1),plot(r ~ nsid,xlab="Participants",xaxt="n",ylab="$p^c_{ik}$",las=1,xlim=c(1,nn),ylim=c(0.05,1.05),cex=.5))
##with(subLL,lines(sid,xcrit,lwd=3)) ## Bayes
with(subLL,lines(sid,xLogit,lty="dashed",lwd=2))
with(subLL,lines(sid,xcritHL,lty="dotted",lwd=5))
abline(v=probSid,lty="dotted")
legend("bottomright",c("LOGIS","COUNT"),lty=c("dashed","dotted"),lwd=c(2,5),bg="white")
aa<-probSid[probSid%%2==0]
axis(1,at=probSid,labels=FALSE)
axis(1,at=aa,tick=FALSE,cex.axis=.45)
aa<-probSid[probSid%%2!=0]
axis(1,at=aa,tick=FALSE,line=.5,cex.axis=.45)
#
par(mar=c(0,4,0,0))
critEstOrd<-within(critEstF[order(critEstF$xcrit),],{
    sid<-reorder(sid,xcrit)
    nsid<-as.numeric(sid)})
with(critEstOrd,plot(tauLogit~nsid,xlim=c(1,nn),ylim=extendrange(critEstOrd$tauLogit),xlab="",xaxt="n",yaxt="n",ylab="\\begin{tabular}{c}LOGIS\\\\$\\beta_1$\\end{tabular}"))
abline(v=probSid,lty="dotted")
axis(2,at=c(-400,0,400),las=1)


## ----mixModels----------------------------------------------------------------
mixModel <- function(dd,type) {
    mer<-lmer(D.contrib ~ L.punrec*xcrit + (1|sidPG) + (1|gidPG),data=dd)
    boot<-bootMer(mer,fixef,bsSize,seed=123,parallel="multicore",ncpus=4)
    ci<-sapply(1:4,function(index) boot.ci(boot,index=index,type=c("norm"))[["normal"]])
    seg<-as.data.frame(t(rbind(ci,fixef(mer)))[,-1])
    names(seg)<-c("low","up","med")
    seg<-within(seg,{i<-1:4;type<-type})
    list(mer=mer,boot=boot,ci=ci,seg=seg)
}
DROP<-mixModel(dd<-within(dataConsist,xcrit<-xLogit),type="DROP")
COUNT<-mixModel(dd<-within(dataAll,xcrit<-xcritHL),type="COUNT")
LOGIS<-mixModel(dd<-within(dataAll,xcrit<-xLogit),type="LOGIS")
betaRange <- function(i) signif(range(sapply(list(DROP,COUNT,LOGIS),function(x) fixef(x$mer)[i])),3)


## ----results='asis'-----------------------------------------------------------
options(xtable.floating=FALSE)
segTab <- function (x,eqn,rownames=TRUE) {
    cat("\\begin{tabular}[t]{r}\\hline\\multicolumn{1}{c}{")
    cat(eqn)
    cat("}\\\\\\hline\n")
    names(x$seg)[1:3] <- c("2.5\\%","97.5\\%","$\\beta$")
    rownames(x$seg)<-c("$\\const$","$\\punI$","$\\probI$","$\\punIprob$")
    print(xtable(x$seg[,c(3,1,2)],digits=3),include.rownames=rownames,sanitize.colnames.function=function(x) x,sanitize.rownames.function=function(x) x)
    cat("\\\\\n")
    xx<-as.data.frame(summary(x$mer)$varcor)
    xx<-cbind(xx,1/ifelse(xx[,4]<1e-13,0,xx[,4]))
    rownames(xx)<-c("$\\nu'_{ik}$","$\\nu_k$","$\\epsilon_{ikt}$")
    colnames(xx)[4:6]<-c("$\\sigma^2$","$\\sigma$","$1/\\sigma^2$")
    print(xtable(xx[,4:6],digits=3),include.rownames=rownames,sanitize.colnames.function=function(x) x,sanitize.rownames.function=function(x) x)
    cat("\\end{tabular}")
}
segTab(DROP,eqn="DROP")
segTab(COUNT,eqn='COUNT',rownames=FALSE)
segTab(LOGIS,eqn='LOGIS',rownames=FALSE)


## -----------------------------------------------------------------------------
beta1range<-signif(range(sapply(c("DROP","COUNT","LOGIS"),function(x) fixef(eval(parse(text=x))[["mer"]]) %*% c(0,1,0,0))),3)
beta13range<-signif(range(sapply(c("DROP","COUNT","LOGIS"),function(x) fixef(eval(parse(text=x))[["mer"]]) %*% c(0,1,0,1))),3)


## ----printMC,results='asis'---------------------------------------------------
printMC <- function (x.mc) {
    options(xtable.floating=FALSE)
    vars<-c("mu[1]","mu[2]","mu[3]","mu[4]","tauS","tauG","tauC")
    x.tab<-cbind(x.mc[["summaries"]][vars,c("Mean","Lower95","Upper95","SSeff","psrf")])
    colnames(x.tab)<-c("Mean","2.5\\%","97.5\\%","SSeff","psrf")
    rownames(x.tab)<-c("$\\const$","$\\punI$","$\\probI$","$\\punIprob$","$\\tau_\\nu$","$\\tau_\\nu'$","$\\tau_\\epsilon$")
    print(xtable(x.tab,digits=c(3,3,3,3,0,4)),sanitize.colnames.function=function(x) x,sanitize.rownames.function=function(x) x)
}
jagsPar <- function(xx,pname="mu[4]",nicename="$\\punIprob$") 
    sprintf(" %.0f chains with %.0f samples each.",
            xx$summary$nchain,xx$sample)


## ----jointModel,echo=FALSE----------------------------------------------------
joint.mod <- 'model {
    # individual choices for risk:
    for (i in 1:length(y)) {
        y[i] ~ dbern(p[i])
        logit(p[i]) <-  (r[i] - 10*xcrit[sid[i]]) * pow(tauCrit[sid[i]],.5)
    }
    # individual choices for contribution:
    for (i in 1:length(D.contrib)) { 
        D.contrib[i] ~ dnorm( mu[1] + mu[2] * L.punrec[i] + 
                      mu[3] * 10*xcrit[sidPG[i]] +
                      mu[4] * 10*xcrit[sidPG[i]] * L.punrec[i] + 
                      nu[sidPG[i]] + nug[gid[sidPG[i]]] , tauC)
    }
    for (s in 1:max(sid)) {               # <- subject specific:
        xcrit[s]   ~ dbeta(ac,bc)         # prior for switching point risk
        tauCrit[s] ~ dgamma(sG,rG)        # var switching point (prior below)
        nu[s] ~ dnorm (0,tauS)             # random effect subject (contrib)
    }
    for (g in 1:max(gid)) {               # <- group specific
        nug[g] ~ dnorm (0,tauG)           # random effect group (contrib)
    }
    ac ~ dgamma(2,1/2)                       # prior for switching point
    bc ~ dgamma(2,1/2)                       # prior for switching point
    sG <- (m*m)/(d*d)                     # prior for var switching point
    rG <- m/(d*d)                         # prior for var switching point
    m ~ dgamma(1,1)                       # prior for var switching point 
    d ~ dgamma(10,1)                       # prior for var switching point 
    tauC ~ dgamma(mTc*mTc/(dTc*dTc),mTc/(dTc*dTc))  # prior var of residual
    mTc ~ dgamma(1,1)
    dTc ~ dgamma(1,1)
    tauS ~ dgamma(mTs*mTs/(dTs*dTs),mTs/(dTs*dTs))  # prior var subj. random effect for beta0
    mTs ~ dgamma(1,1)
    dTs ~ dgamma(1,1)
    tauG ~ dgamma(mTg*mTg/(dTg*dTg),mTg/(dTg*dTg))  # prior var group random effect for beta0
    mTg ~ dgamma(1,1)
    dTg ~ dgamma(1,1)
    for (k in 1:4) {
       mu[k]   ~ dnorm (0,.0001)          # prior fixed effect for beta
    }
}'






## ----figEstSwitchPointB,fig.height=2.5,fig.width=6.01-------------------------
nn<-length(levels(subLL$sid))
layout(rbind(2,3,1),heights=c(.25,.05,.7))
#    
par(mar=c(4,4,0,0))
with(subset(subLL,y==1),plot(r ~ nsid,xlab="Participants",xaxt="n",ylab="$p^c_{ik}$",las=1,xlim=c(1,nn),ylim=c(0.05,1.05),cex=.5))
with(subLL,lines(sid,xcrit,lwd=3))
with(subLL,lines(sid,xLogit,lty="dashed",lwd=2))
with(subLL,lines(sid,xcritHL,lty="dotted",lwd=5))
abline(v=probSid,lty="dotted")
legend("bottomright",c("B-JOINT","LOGIS","COUNT"),lty=c("solid","dashed","dotted"),lwd=c(3,2,5),bg="white")
aa<-probSid[probSid%%2==0]
axis(1,at=probSid,labels=FALSE)
axis(1,at=aa,tick=FALSE,cex.axis=.45)
aa<-probSid[probSid%%2!=0]
axis(1,at=aa,tick=FALSE,line=.5,cex.axis=.45)
#
par(mar=c(0,4,.01,0))
critEstOrd<-within(critEstF[order(critEstF$xcrit),],{
    sid<-reorder(sid,xcrit)
    nsid<-as.numeric(sid)})
with(critEstOrd,plot(1/sqrt(tau) ~ nsid,xlim=c(1,nn),xlab="",xaxt="n",las=1,ylab="$\\sigma_{ik}(\\probcrit)$"))
abline(v=probSid,lty="dotted")
##axis(2,at=seq(.2,.4,.2),las=1)
##axis(2,at=seq(.2,.4,.2),labels=FALSE)
#


## ----jjSummaries,cache=TRUE---------------------------------------------------
load("data/jj.Rdata")


## ----tabjjAll,results='asis'--------------------------------------------------
printMC(jjAll.jags)


## ----tabjjConsist,results='asis'----------------------------------------------
printMC(jjConsist.jags)


## ----interActRange1-----------------------------------------------------------
interActRange1<-range(fixef(DROP[["mer"]])[4],fixef(COUNT[["mer"]])[4],fixef(LOGIS[["mer"]])[4])
intJC<-jjConsist.jags[["summary"]][["statistics"]]["mu[4]","Mean"]
intJA<-jjAll.jags[["summary"]][["statistics"]]["mu[4]","Mean"]


## ----segplotBeta,fig.width=6,cache=FALSE--------------------------------------
# paste the two estimates together:
mc2seg<-function(mc,type) {
    x<-within(as.data.frame(mc$summary$quantiles[c("mu[1]","mu[2]","mu[3]","mu[4]"),c("2.5%","50%","97.5%")]),{i<-1:4;type<-type})
    rownames(x)<-NULL
    names(x)[1:3]<-c("low","med","up")
    x
    }
xx<-as.data.frame(rbind(mc2seg(jjAll.jags,"B-JOINT"),mc2seg(jjConsist.jags,"B-JOINT-DROP")))
# merge ME model:
xx<-rbind.fill(xx,rbind.fill(lapply(list(DROP,COUNT,LOGIS),function(x) x$seg)))
# give them nice labels:
bCoefs<-c("const","punI","probI","punIprob")
xx<-within(xx,{beta<-reorder(factor(sprintf("$\\beta_{\\%s}$",bCoefs[i])),i);
    type<-factor(type)})
# plot HPD intervals:
segplot( type~low+up | beta,data=xx,centers=med,horizontal=TRUE,
        scales=list(x=list(relation="free")),draw.bands=FALSE,
        segments.fun=panel.arrows,ends="both",angle=90,length=.1,
        layout=c(4,1))+layer(panel.abline(v=0))


## ----saveImage----------------------------------------------------------------
save.image("image.Rdata")


## ----simul2,eval=FALSE--------------------------------------------------------
## aSim <- function(i) {
##     set.seed(i)
##     muSim<-rnorm(4,0,2)
##     N<-100;gSize<-4;T<-10
##     sidPG<-rep(1:N,each=T)
##     gidSim<-(0:(N-1)) %/% 4 + 1
##     gidPG<-rep(gidSim,each=T)
##     nu<-rnorm(max(gidSim),0,sqrt(1/5))[gidSim]
##     nup<-rep(rnorm(N,0,sqrt(1/3.5)),each=T)
##     eps<-rnorm(N*T,0,sqrt(1/sqrt(.1)))
## #
##     xcritSim<-10*rbeta(N,6.98,3.63)
##     xcritSimT<-rep(xcritSim,each=T)
##     tauSim<-rgamma(N,0.847,.2)
##     L.punrec<- floor(rgamma(N*T,.3,.2))
## # choices in HL task:
##     xc<-rbind.fill(lapply(1:N,function(i) {
##         y<-rbinom(10,1,plogis(((1:10)-xcritSim[i])*sqrt(tauSim[i])))
##         consist<-sum(y[1:(10-sum(y))])==0
##         xcrit<-10-sum(y)+1/2
##         as.data.frame(list(sid=i,y=y,r=1:10,consist=consist,xcrit=xcrit)) }))
## 
##     mData<-list()
##     mData[["y"]]<-xc$y
##     mData[["r"]]<-xc$r
##     mData[["sid"]]<-xc$sid
##     mData[["sidPG"]]<-xc$sid
##     mData[["D.contrib"]]<-c(cbind(1,L.punrec,xcritSimT,L.punrec*xcritSimT) %*% cbind(muSim)) + nu + nup + eps
##     mData[["L.punrec"]]<-L.punrec
##     mData[["xLogit"]]<-xc$xcrit
##     mData[["gid"]]<-gidSim
##     mData[["gidPG"]]<-gidPG
##     #
##     sim.fe<-fixef(with(mData,lmer(D.contrib ~ L.punrec*xLogit + (1|sidPG) + (1|gidPG))))
##     runjags.options(silent.jags=TRUE,silent.runjags=TRUE,method="parallel",modules="glm",predraw.plots=FALSE)
##     system.time(jjAll.jags<-run.jags(model=joint.mod,data=mData,monitor=c("mu","ac","bc"),n.chains=4,inits=initJags))
##     list(mu=muSim,ec.sum=summary(jjAll.jags),me.fe=sim.fe)
## }
## sim.list<-list()
## system.time(sim.list[[1]]<-aSim(1))
## system.time(for (i in 2:100) {
##     cat(i,"\n")
##     t<-system.time(sim.list[[i]]<-aSim(i))
##     save(sim.list,file="sim2N.Rdata")
##     cat(t)
## })

## ----simul3,cache=FALSE-------------------------------------------------------
load("sim2N.Rdata")


## ----fig:muHat,cache=FALSE----------------------------------------------------
me4<-unlist(lapply(sim.list,function(l) l[["mu"]][4]))
ec.sum<-unlist(lapply(sim.list,function(l) l[["ec.sum"]][["quantiles"]]["mu[4]","50%"]))
me.fe<-unlist(lapply(sim.list,function(l) l[["me.fe"]][4]))
underest<-signif(100*(1-coef(lm(me.fe ~ me4))[2]),3)
xx<-rbind(as.data.frame(list(muHat=ec.sum,mu=me4,type="B-JOINT")),as.data.frame(list(muHat=me.fe,mu=me4,type="COUNT")))
library(splines)
library("quantreg")
#
nSpline<-5
xx.quant<-rbind.fill(by(xx,list(xx$type),function(x) {
    X<-with(x,model.matrix(muHat-mu ~ bs(mu,df=nSpline)))
    ldply(list(.25,.5,.75),function(tau) data.frame(mu=x$mu,bias=X %*% coef(rq(muHat-mu ~ bs(mu,df=nSpline),tau=tau,data=x)),tau=tau,type=x$type))}))
xx.quant<-within(xx.quant,tauF<-reorder(factor(sprintf("%g\\%%",100*tau)),-tau))
xyplot(bias ~ mu | type,group=tauF,data=xx.quant,t="a",xlab="$\\beta_\\punIprob$",ylab="$\\hat{\\beta}_\\punIprob-\\beta_\\punIprob$",auto.key=list(points=FALSE,lines=TRUE,space="right",size=3,title="Quantiles:",cex.title=1),)+layer(panel.refline(h=0)) 


## ----figAlphaGamma,fig.width=3.2----------------------------------------------
xxNullT<-within(stack(data.frame(xxNull[sample(1:nrow(xxNull),1000),c("sG","rG")])),type<-"prior")
xxT <-within(stack(dfSample(as.data.frame(xxCrit[,c("sG","rG")]),1000)),type<-"posterior")
xxTNullT<-within(rbind(xxNullT,xxT),{
                 ind<-relevel(ind,"sG");
                 levels(ind)<-c("$\\alpha_\\tau$","$\\beta_\\tau$")
                 it<-reorder(as.factor(sprintf("%s %s",type,ind)),values,FUN=median)
                 })
ecdfplot(~values ,group=it,data=xxTNullT,xlim=exp(quantile(log(xxTNullT$values),c(.001,.999))),scales=list(x=list(log=10)),xscale.components=xscale.components.log,auto.key=list(columns=2,between=1,size=2),xlab="$\\alpha_\\tau,\\beta_\\tau$")


## ----figXCrit,fig.width=3.2---------------------------------------------------
xxNullC<-within(stack(simCols(xxNull,c("xcrit"),2000)),type<-"prior")
xxC<-within(stack(simCols(xxCrit,c("xcrit"),2000)),type<-"posterior")
xxCNullC<-within(rbind(xxNullC,xxC),{
    levels(ind)<-c("$\\probcrit_{ik}$")
    type<-reorder(type,values,FUN=median)
    })
ecdfplot(~values ,group=type,data=xxCNullC,auto.key=list(columns=2,between=1,size=2),xlab="$\\probcrit_{ik}$")


## ----tauEcdf------------------------------------------------------------------
tauAll<-rbind.fill(as.data.frame(list(tau=sample(c(xxTauCrit),1000),type="posterior $\\sigma_{ik}$")),
   as.data.frame(list(tau=xxCritMedian[grep("tauCrit",names(xxCritMedian))],type="median $\\sigma_{ik}$ per participant")))
ecdfplot(~ 1/sqrt(tau),group=type,data=tauAll,auto.key=list(columns=2),xlim=c(0,quantile(1/sqrt(tauAll$tau),c(.98))),xlab="$\\sigma_{ik}$",scales=list(y=list(draw=FALSE)))+layer(panel.abline(h=consistentShare,v=(1/sqrt(tauConsist)),lty=3))


## ----echo=FALSE,cache=FALSE---------------------------------------------------
opts_chunk[["set"]](echo=TRUE)
library(runjags)
runjags.options(silent.jags=TRUE, silent.runjags=TRUE, predraw.plots=FALSE)
library(coda)
jagsVer<-testjags(silent=TRUE)
rjver<-sessionInfo()[["otherPkgs"]][["runjags"]][["Version"]]


## ----loadLib------------------------------------------------------------------
library(runjags)


## ----loadPlyr-----------------------------------------------------------------
library(plyr)


## ----setSeed------------------------------------------------------------------
set.seed(1)


## ----setN---------------------------------------------------------------------
N<-40


## ----data.A-------------------------------------------------------------------
choiceNum.A<-10                                ## number of choices in A
sid.A<-1:N                                     ## subject IDs
thresh.A<-sort(choiceNum.A*rbeta(N,6.98,3.63)) ## (latent) threshold for 
                         ##each subject; here we sort the individuals for convenience
tau.A<-rgamma(N,0.847,.2)         ## (latent) precision of each individual for task A
data.A<-rbind.fill(lapply(1:N,function(sid) {
      choice.A<-rbinom(choiceNum.A,1,          ## observable choices
                       plogis(((1:choiceNum.A)-thresh.A[sid])*sqrt(tau.A[sid])))
      data.frame(list(sid=sid,choice.A=choice.A,task.A=1:10)) }))


## ----plot.A,echo=1:2----------------------------------------------------------
with(subset(data.A,choice.A==1),plot(sid,task.A))
lines(sid.A,thresh.A)
xx<-merge(aggregate(cbind(min=task.A) ~ sid,data=subset(data.A,choice.A==1),FUN=min),
aggregate(cbind(max=task.A) ~ sid,data=subset(data.A,choice.A==0),FUN=max))
with(subset(xx,min<max),abline(v=sid,col="red",lty="dotted"))


## ----data.B.small-------------------------------------------------------------
sid.B<-1:N
x1.B <- runif(N) ## control variable
x2.B <- runif(N) ## control variable
eps.B<- rnorm(N,0,sqrt(1/sqrt(.1))) ## residual
X.B  <- cbind(1,x1.B,x2.B,thresh.A) ## design matrix
beta <- rbind(-1,1,0,-1)         ## arbitrary coefficients
y.B  <- c(X.B %*% beta + eps.B)


## ----prepareDataForJags-------------------------------------------------------
sData<-as.list(data.A)
sData[["choiceNum.A"]] <- choiceNum.A
sData[["y.B"]]<-y.B
sData[["X.B"]]<-X.B       ## demeaning does not help convergence here
sData[["K"]] <- ncol(X.B) ## easier to determine size of beta here 
sData[["latentA"]] <- 4   ## latent variable is 4th column
sData[["X.B"]][,4]<-NA    ## we can't observe the latent variable


## ----smallModel,results="asis",echo=FALSE-------------------------------------
small.mod <- "model {
    # binary choices in A:
    for (i in 1:length(choice.A)) {
        choice.A[i] ~ dbern(p[i])
        logit(p[i]) <-  (task.A[i] - choiceNum.A*thresh.A[sid[i]]) * 
                                       pow(tau.A[sid[i]],.5)
    }
    # individual choices in B (here one choice per individual):
    for (i in 1:length(y.B)) { 
        y.B[i] ~ dnorm(inprod(X.B[i,],beta),tau.B)
        X.B[i,latentA] <- choiceNum.A*thresh.A[i] # estimate for latent var.
    }
    # subject specific:
    for (s in 1:max(sid)) {
        thresh.A[s] ~ dbeta(a.A,b.A) # switching point
        tau.A[s]    ~ dgamma(s.A,r.A)   # precision switching point
    }
    # parameters for switching point:
    a.A ~ dgamma(2,1/2)          # switching point
    b.A ~ dgamma(2,1/2)          # switching point
    s.A <- (m.A*m.A)/(d.A*d.A)   # prec. switching point
    r.A <- m.A/(d.A*d.A)         # prec. switching point
    m.A ~ dgamma(1,1)            # prec. switching point 
    d.A ~ dgamma(10,1)           # prec. switching point 
    ## residuals:
    tau.B ~ dgamma(mTB*mTB/(dTB*dTB),mTB/(dTB*dTB))
    mTB ~ dgamma(1,1)
    dTB ~ dgamma(1,1)
    ## coefficients:
    for (k in 1:K) {
       beta[k] ~ dnorm (0,.0001) # prior fixed effect for beta
    }
}"
cat("small.mod <- '",small.mod,"'",sep="")


## ----jagsIni,cache=TRUE-------------------------------------------------------
runjags.options(silent.jags=TRUE,silent.runjags=TRUE,method="parallel",
                modules="glm",predraw.plots=FALSE)
initJags<-list()
initJags[[1]]<-list(.RNG.seed=1,.RNG.name="base::Mersenne-Twister")
initJags[[2]]<-list(.RNG.seed=2,.RNG.name="base::Super-Duper")
initJags[[3]]<-list(.RNG.seed=3,.RNG.name="base::Wichmann-Hill")
initJags[[4]]<-list(.RNG.seed=4,.RNG.name="lecuyer::RngStream")


## ----jagsThresh,cache=TRUE----------------------------------------------------
thresh.jags<-run.jags(model=small.mod,within(sData,{y.B<-NA;choice.A=NA}),
                      monitor=c("thresh.A"),n.chains=4,inits=initJags,sample=100)
thresh.df<-data.frame(as.mcmc(thresh.jags))


## ----jagsThreshPlot,echo=TRUE,eval=FALSE--------------------------------------
## plot(ecdf(unlist(thresh.df)))

## ----jagsThreshPlot2,echo=FALSE-----------------------------------------------
par(mar=c(4,4,1,.1))
plot(ecdf(sample(unlist(thresh.df),1000)))


## ----jagsSmall,cache=TRUE-----------------------------------------------------
run.jags(model=small.mod,data=sData,monitor=c("beta"),n.chains=4,inits=initJags)


## ----data.B.large-------------------------------------------------------------
groups.B    <-4
choiceNum.B <-2
sid.B<-rep(1:N,each=choiceNum.B) ## subject ID
gid.B<-sid.B %% groups.B + 1     ## group ID
x1.B <- runif(N*choiceNum.B)
x2.B <- runif(N*choiceNum.B)
eps.B<- rnorm(N*choiceNum.B,0,sqrt(1/sqrt(.1))) ## residuals
nu1.B<- rnorm(groups.B,0,sqrt(1/5))[gid.B]## random effect for each group
nu2.B<- rnorm(N,0,sqrt(1/3.5))[sid.B]     ## random effect for each subject
X.B  <- cbind(1,x1.B,x2.B,thresh.A)       ## design matrix
beta <- rbind(-1,1,0,-1)         ## arbitrary coefficients
y.B  <- c(X.B %*% beta + nu1.B + nu2.B + eps.B)


## ----prepareDataForJags.large-------------------------------------------------
lData<-as.list(data.A)
lData[["choiceNum.A"]] <- choiceNum.A
lData[["sid.B"]]<-sid.B
lData[["gid.B"]]<-gid.B
lData[["y.B"]]<-y.B
lData[["X.B"]]<-X.B ## demeaning X.B is not necessary here
lData[["K"]] <- ncol(X.B) ## easier to determine size of beta here 
lData[["latentA"]] <- 4   ## latent variable is 4th column
lData[["X.B"]][,4]<-NA    ## we can't observe the latent variable


## ----largerModel,results="asis",echo=FALSE------------------------------------
large.mod <- 'model {
    # binary choices in A:
    for (i in 1:length(choice.A)) {
        choice.A[i] ~ dbern(p[i])
        logit(p[i]) <-  (task.A[i] - choiceNum.A*thresh.A[sid[i]]) * 
                                       pow(tau.A[sid[i]],.5)
    }
    # individual choices in B:
    for (i in 1:length(y.B)) { 
        y.B[i] ~ dnorm(inprod(X.B[i,],beta) + nu1[gid.B[i]] + nu2[sid.B[i]],tau[3])
                                             ## nu1, nu2 are the two random effects
        X.B[i,latentA] <- choiceNum.A*thresh.A[sid.B[i]] # estimate for latent var.
    }
    # subject specific:
    for (s in 1:max(sid)) {
        thresh.A[s] ~ dbeta(a.A,b.A)    # switching point for A
        tau.A[s]    ~ dgamma(s.A,r.A)   # precision switching point
        nu2[s]      ~ dnorm (0,tau[2])  # random effect subject for B
    }
    # group specific
    for (g in 1:max(gid.B)) {
        nu1[g] ~ dnorm (0,tau[1])        # random effect group for B 
    }
    # parameters for switching point:
    a.A ~ dgamma(2,1/2)          # switching point
    b.A ~ dgamma(2,1/2)          # switching point
    s.A <- (m.A*m.A)/(d.A*d.A)   # prec. switching point
    r.A <- m.A/(d.A*d.A)         # prec. switching point
    m.A ~ dgamma(1,1)            # prec. switching point 
    d.A ~ dgamma(10,1)           # prec. switching point 
    ##
    for (k in 1:3) {  ## priors for prec. of residual and random effects:
       tau[k] ~ dgamma(mTB[k]*mTB[k]/(dTB[k]*dTB[k]),mTB[k]/(dTB[k]*dTB[k]))
       mTB[k] ~ dgamma(1,1)
       dTB[k] ~ dgamma(1,1)
    }
    for (k in 1:K) {
       beta[k] ~ dnorm (0,.0001) # prior fixed effect for beta
    }
}'
cat("large.mod <- '",large.mod,"'",sep="")


## ----jagsLarge,cache=TRUE-----------------------------------------------------
run.jags(model=large.mod,data=lData,monitor=c("beta"),n.chains=4,inits=initJags)


## ----echo=FALSE---------------------------------------------------------------
opts_chunk[["set"]](echo=FALSE)
par(mar=c(4,4,2,.1))
q<-c(13.2,14,14.2,14.7,14.2,14.2,14.5,14.3,14,13.7)
plot(NULL,ylim=c(0,20),xlim=c(1,10),ylab="average contribution",main="past experiment",xlab="period",type=c("b"),pch=2,yaxp=c(0,20,10))
abline(h=seq(0,20,2),lty=3)
lines(1:10,q)
points(1:10,q,pch=2)

## ----echo=FALSE---------------------------------------------------------------
#save.image("all.Rdata")

